//-----------------------------------------------------------------------------
// Microsoft OLE DB RowsetViewer
// Copyright (C) 1994 - 1998 By Microsoft Corporation.
//
// @doc
//
// @module COMMON.CPP
//
//-----------------------------------------------------------------------------------

/////////////////////////////////////////////////////////////////////////////
// Include
//
/////////////////////////////////////////////////////////////////////////////
#include "common.h"


/////////////////////////////////////////////////////////////////////////////
// HRESULT ConvertToMBCS
//
/////////////////////////////////////////////////////////////////////////////
HRESULT ConvertToMBCS(WCHAR* pwsz, CHAR* psz, ULONG cStrLen)
{
	ASSERT(psz);

	//No-op
	if(pwsz==NULL)
	{
		psz[0] = EOL;
		return S_FALSE;
	}

	//Convert the string to MBCS
	INT iResult = WideCharToMultiByte(CP_ACP, 0, pwsz, -1, psz, cStrLen, NULL, NULL);
	
	//Add NULL Terminator
	psz[min((ULONG)iResult, cStrLen-1)] = EOL;
	return iResult ? S_OK : E_FAIL;
}

/////////////////////////////////////////////////////////////////////////////
// HRESULT ConvertToMBCS
//		Dynamically allocated memory
/////////////////////////////////////////////////////////////////////////////
CHAR* ConvertToMBCS(WCHAR* pwsz)
{
	//no-op case
	if(!pwsz)
		return NULL;
	
	//Determine the space required for the conversion
	LONG iLen = WideCharToMultiByte(CP_ACP, 0, pwsz, -1, NULL, 0, NULL, NULL);

	//Allocate space for the string
	CHAR* pszBuffer = NULL;
	SAFE_ALLOC(pszBuffer, CHAR, iLen+1);

	//Now convert the string
	WideCharToMultiByte(CP_ACP, 0, pwsz, -1, pszBuffer, iLen+1, NULL, NULL);

CLEANUP:
	return pszBuffer;
}


/////////////////////////////////////////////////////////////////////////////
// HRESULT ConvertToWCHAR
//
/////////////////////////////////////////////////////////////////////////////
HRESULT ConvertToWCHAR(CHAR* psz, WCHAR* pwsz, ULONG cStrLen)
{
	ASSERT(pwsz);

	//No-op
	if(psz==NULL)
	{
		pwsz[0] = wEOL;
		return S_FALSE;
	}

	//Convert the string to MBCS
	INT iResult = MultiByteToWideChar(CP_ACP, 0, psz, -1, pwsz, cStrLen);

	//Add NULL Terminator
	pwsz[min((ULONG)iResult, cStrLen-1)] = wEOL;
	return iResult ? S_OK : E_FAIL;
}



/////////////////////////////////////////////////////////////////////////////
// HRESULT ConvertToWCHAR
//		Dynamically allocated memory
/////////////////////////////////////////////////////////////////////////////
WCHAR* ConvertToWCHAR(CHAR* psz)
{
	//no-op case
	if(!psz)
		return NULL;
	
	//Determine the space required for the conversion
	LONG iLen = MultiByteToWideChar(CP_ACP, 0, psz, -1, NULL, 0);

	//Allocate space for the string
	WCHAR* pwszBuffer = NULL;
	SAFE_ALLOC(pwszBuffer, WCHAR, iLen+1);

	//Now convert the string
	MultiByteToWideChar(CP_ACP, 0, psz, -1, pwszBuffer, iLen+1);

CLEANUP:
	return pwszBuffer;
}



/////////////////////////////////////////////////////////////////////////////
// HRESULT GUIDFromString
//
/////////////////////////////////////////////////////////////////////////////
HRESULT GUIDFromString(CHAR* psz, GUID* pGuid)
{
	if(psz == NULL || pGuid == NULL)
		return E_FAIL;

	WCHAR* pwsz = ConvertToWCHAR(psz);
	HRESULT hr = CLSIDFromString(pwsz, pGuid);

	SAFE_FREE(pwsz);
	return hr;
}


/////////////////////////////////////////////////////////////////////////////
// HRESULT ConvertString
//
/////////////////////////////////////////////////////////////////////////////
HRESULT ConvertString(CHAR* psz, ULONG ulMaxSize, DWORD dwFlags)
{
	ASSERT(psz);
	ASSERT(ulMaxSize);
	CHAR* pszStop = NULL;

	//Currently only deals with HEX or DECIMAL
	ASSERT(dwFlags & CONV_HEX || dwFlags & CONV_DECIMAL);

	//Convert String to integer
	LONG lValue = strtol(psz, &pszStop, 10);
	if(lValue == LONG_MAX || lValue == 0 || pszStop==NULL || pszStop[0]!=EOL)
		return E_FAIL;

	//Convert Integer back to requested Format
	if(dwFlags & CONV_HEX)
		sprintf(psz, "0x%x", lValue);
	else
		sprintf(psz, "%d", lValue);

	return S_OK;
}


/////////////////////////////////////////////////////////////////////////////
// HRESULT BYTESToString
//
/////////////////////////////////////////////////////////////////////////////
HRESULT BYTESToString(BYTE* pb, CHAR* pszBuffer, ULONG cBytes, ULONG* pulStrLen)
{
	ASSERT(pb && pszBuffer);

	// Process the byte stream, converting each byte into
	// its two character unicode hexaDECIMAL representation
	static CHAR s_chHex[] = 
	{'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};

	for(ULONG i=0; i<cBytes; i++)
	{
		*pszBuffer++ = s_chHex[*pb >> 4];
		*pszBuffer++ = s_chHex[*pb & 0x0F];
		pb++;
	}
	*pszBuffer = EOL;
	
	if(pulStrLen)
		*pulStrLen = cBytes << 1;
	return S_OK;
}


/////////////////////////////////////////////////////////////////////////////
// HRESULT StringToBYTES
//
/////////////////////////////////////////////////////////////////////////////
HRESULT StringToBYTES(CHAR* pszBuffer, BYTE* pb, ULONG cBytes, ULONG* pcbBytes)
{
	ASSERT(pszBuffer && pb);

	// Process the byte stream, converting each byte into
	// its two character unicode hexaDECIMAL representation
	static CHAR s_chByte[] = 
	{'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', '?', '?', '?', '?', '?', '?', '?', 'A', 'B', 'C', 'D', 'E', 'F'};

	for(ULONG i=0; i<cBytes; i++)
	{
		// Process first character of pair
		CHAR ch = *pszBuffer++;
		if( ch >= '0' && ch <= '9' )
			ch -= '0';
		else if( ch >= 'A' && ch <= 'F' )
			ch -= 'A' - 10;
		else if( ch >= 'a' && ch <= 'f' )
			ch -= 'a' - 10;
		else
			return DB_E_CANTCONVERTVALUE;
		*pb = (BYTE)ch << 4;

		// Advance to second character of pair
		ch = *pszBuffer++;
		if( ch >= '0' && ch <= '9' )
			ch -= '0';
		else if( ch >= 'A' && ch <= 'F' )
			ch -= 'A' - 10;
		else if( ch >= 'a' && ch <= 'f' )
			ch -= 'a' - 10;
		else
			return DB_E_CANTCONVERTVALUE;
		*pb |= (BYTE)ch;

		// Done two characters -- created one byte
		pb++;
	}
	
	if(pcbBytes)
		*pcbBytes = cBytes;
	return S_OK;
}


/////////////////////////////////////////////////////////////////////////////
// HRESULT VariantToString
//
/////////////////////////////////////////////////////////////////////////////
HRESULT VariantToString(VARIANT* pVariant, CHAR* psz, ULONG ulMaxSize, DWORD dwFlags)
{
	//Convert a VARIANT to a WCHAR
	ASSERT(pVariant);
	ASSERT(psz);
	ASSERT(ulMaxSize > 0);
	HRESULT hr = S_OK;
	
	//Find the VariantType
	DBTYPE wType = V_VT(pVariant);
	VARIANT VarTemp;
	VariantInit(&VarTemp);
	static LCID lcid = GetSystemDefaultLCID();

	//VT_ARRAY is not handled by VariantChangeTypeEx
	if(wType & VT_ARRAY)
	{
		TESTC(hr = SafeArrayToString(V_ARRAY(pVariant), wType, psz, ulMaxSize));
		goto CLEANUP;
	}

	switch(wType)
	{
		case VT_NULL:
		case VT_EMPTY:
			psz[0] = EOL;
			break;

		case VT_BOOL:
			if(V_BOOL(pVariant) == VARIANT_TRUE)
				sprintf(psz, "%s", dwFlags & CONV_VARBOOL ? "VARIANT_TRUE" : "True");
			else if(V_BOOL(pVariant) == VARIANT_FALSE)
				sprintf(psz, "%s", dwFlags & CONV_VARBOOL ? "VARIANT_FALSE" : "False");
			else
				sprintf(psz, "%d", V_BOOL(pVariant));
			break;

		case VT_ERROR:
			sprintf(psz, "%d", V_ERROR(pVariant));
			break;

		default:
		{
			//Delgate
			TESTC(hr = VariantChangeTypeEx(
				&VarTemp,	// Destination (convert not in place)
				pVariant,	// Source
				lcid,		// LCID
				0,			// dwFlags
				VT_BSTR ));
	
			//Convert to MBCS
			ConvertToMBCS(V_BSTR(&VarTemp), psz, ulMaxSize);
		}
	};

CLEANUP:
	psz[ulMaxSize-1] = EOL;
	XTEST(NULL, VariantClear(&VarTemp));
	return hr;
}


/////////////////////////////////////////////////////////////////////////////
// HRESULT VariantToString
//
/////////////////////////////////////////////////////////////////////////////
HRESULT VariantToString(VARIANT* pVariant, WCHAR* pwsz, ULONG ulMaxSize, DWORD dwFlags)
{
	//Convert a VARIANT to a WCHAR
	ASSERT(pVariant);
	ASSERT(pwsz);
	ASSERT(ulMaxSize > 0);
	HRESULT hr = S_OK;
	
	//Find the VariantType
	DBTYPE wType = V_VT(pVariant);
	VARIANT VarTemp;
	VariantInit(&VarTemp);
	static LCID lcid = GetSystemDefaultLCID();

	//VT_ARRAY is not handled by VariantChangeTypeEx
	if(wType & VT_ARRAY)
	{
		CHAR szBuffer[MAX_COL_SIZE];
		ASSERT(ulMaxSize >= MAX_COL_SIZE);
		TESTC(hr = SafeArrayToString(V_ARRAY(pVariant), wType, szBuffer, MAX_COL_SIZE));
		ConvertToWCHAR(szBuffer, pwsz, MAX_COL_SIZE);
		goto CLEANUP;
	}

	switch(wType)
	{
		case VT_NULL:
		case VT_EMPTY:
			pwsz[0] = EOL;
			break;

		case VT_BOOL:
			if(V_BOOL(pVariant) == VARIANT_TRUE)
				swprintf(pwsz, L"%s", dwFlags & CONV_VARBOOL ? L"VARIANT_TRUE" : L"True");
			else if(V_BOOL(pVariant) == VARIANT_FALSE)
				swprintf(pwsz, L"%s", dwFlags & CONV_VARBOOL ? L"VARIANT_FALSE" : L"False");
			else
				swprintf(pwsz, L"%d", V_BOOL(pVariant));
			break;

		case VT_ERROR:
			swprintf(pwsz, L"%d", V_ERROR(pVariant));
			break;

		default:
		{
			TESTC(hr = VariantChangeTypeEx(
				&VarTemp,	// Destination (convert not in place)
				pVariant,	// Source
				lcid,		// LCID
				0,			// dwFlags
				VT_BSTR ));

			//Convert to MBCS
			wcsncpy(pwsz, V_BSTR(&VarTemp), ulMaxSize);

			//May need Hex postprocessing
			if(dwFlags & CONV_HEX)
			{
				switch(V_VT(pVariant))
				{
					case DBTYPE_I4:
						swprintf(pwsz, L"0x%x", V_I4(pVariant));
						break;
				}
			}
		}
	};

CLEANUP:
	pwsz[ulMaxSize-1] = wEOL;
	XTEST(NULL, VariantClear(&VarTemp));
	return hr;
}


/////////////////////////////////////////////////////////////////////////////
// HRESULT StringToVariant
//
/////////////////////////////////////////////////////////////////////////////
HRESULT StringToVariant(WCHAR* pwsz, VARTYPE vt, VARIANT* pVariant, DWORD dwFlags)
{
	//Convert a VARIANT to a WCHAR
	ASSERT(pwsz);
	ASSERT(pVariant);
	HRESULT hr = S_OK;

	//Assign the type...
	V_VT(pVariant) = vt;
	static LCID lcid = GetSystemDefaultLCID();

	//VariantChangeTypeEx seems to handle most types,
	//except the following cases...
	switch(vt)
	{
		case VT_NULL:
		case VT_EMPTY:
			break;

		case VT_BOOL:
		{	
			if(dwFlags & CONV_VARBOOL && _wcsicmp(pwsz, L"VARIANT_TRUE")==0)
				V_BOOL(pVariant) = VARIANT_TRUE;
			else if(dwFlags & CONV_VARBOOL && _wcsicmp(pwsz, L"VARIANT_FALSE")==0)
				V_BOOL(pVariant) = VARIANT_FALSE;
			else if(dwFlags & CONV_ALPHABOOL && _wcsicmp(pwsz, L"True")==0)
				V_BOOL(pVariant) = VARIANT_TRUE;
			else if(dwFlags & CONV_ALPHABOOL && _wcsicmp(pwsz, L"False")==0)
				V_BOOL(pVariant) = VARIANT_FALSE;
			else
			{
				LONG lValue = wcstol(pwsz, NULL, 0);
				if(lValue == LONG_MAX || lValue>SHRT_MAX || lValue<SHRT_MIN)
				{
					hr = E_FAIL;
					goto CLEANUP;
				}
				V_BOOL(pVariant) = (VARIANT_BOOL)lValue;
			}
			break;
		}

		case VT_I4:
		case VT_UI4:
		case VT_ERROR:
		{	
			//We handle this case seperatly since we want to handle HEX values
			WCHAR* pwszStop = NULL;
			LONG ulValue = wcstol(pwsz, &pwszStop, 0);
			if(pwszStop==NULL || pwszStop[0]!=wEOL)
			{
				hr = E_FAIL;
				goto CLEANUP;
			}

			V_I4(pVariant) = ulValue;
			break;
		}

		default:
		{
			//Place the string into the BSTR of the VARIANT
			V_VT(pVariant) = VT_BSTR;
			V_BSTR(pVariant) = SysAllocString(pwsz);
			
			//Now delegate to VariantChangeType...
			TESTC(hr = VariantChangeTypeEx(
				pVariant,	// Destination (convert in place)
				pVariant,	// Source
				lcid,		// LCID
				0,			// dwFlags
				vt ));
		}
	};

CLEANUP:
	return hr;
}


/////////////////////////////////////////////////////////////////////////////
// HRESULT SafeArrayToString
//
/////////////////////////////////////////////////////////////////////////////
HRESULT SafeArrayToString(SAFEARRAY* pSafeArray, DBTYPE wType, CHAR* pszBuffer, ULONG ulMaxSize)
{
	ASSERT(pSafeArray);

	//This method is capable of handling N-Dimenstions of Data!!
	//We need to take N-Dimensions of Data and convert it to a string

	//For example:  We have a 3D data, where z-dim has 2 elements, y-Dim has 3 ele
	//and x-dim has 2 elements we end up with the following matrix of points
	
	//	(z, y, x)  => value	[[[
	//	(1, 1, 1)  => 1
	//	(1, 1, 2)  => 2
	//	(1, 2, 1)  => 3		][
	//	(1, 2, 2)  => 4
	//	(1, 3, 1)  => 5		][
	//	(1, 3, 2)  => 6
	//	(2, 1, 1)  => 7		]][[
	//	(2, 1, 2)  => 8
	//	(2, 2, 1)  => 9		][
	//	(2, 2, 2)  => A
	//	(2, 3, 1)  => B		][
	//	(2, 3, 2)  => C
	//						]]]

	//So what we need to generate is a string of:
	//	[ [[1,2][3,4][5,6]] [[7,8][9,A][B,C]] ]

	//The algorythm we are using is bascially based upon a "ripple" counter.
	//We keep a counter for each dimension.  So when CounterDim[0] hits the Upper
	//Limit, we increment CounterDim[1] and set CounterDim[0] back to LowerLimit.  
	//This continues until all have reached the upper limit together:
	//{CounterDim[n-1]==UpperLimt, ... Dim[0]==UpperLimit}

	//The braces are determined by rising/falling edges of the ripple counter.
	//Everytime a dimension restarts its value from Upper->Lower we see a "][".
	//So we have a pre and a post set of - "[[[" braces "]]]" for the number of dimensions.
	//You'll notices the set of braces in the above example on the rising/falling
	//edges of the ripple counter....

	HRESULT hr = S_OK;

	VARIANT Variant;
	VariantInit(&Variant);
	CHAR* psz = pszBuffer;

	ULONG i,iDim, ulInnerElements = 0;
	ULONG cDimensions = SafeArrayGetDim(pSafeArray);
	BOOL bDone = FALSE;

	//No-op
	if(cDimensions < 1)
		return E_FAIL;

	//Make sure there is no Array in the type
	wType &= ~VT_ARRAY;

	ULONG ulStrLen, ulSizeLeft = ulMaxSize;
	pszBuffer[0] = EOL;

	LONG* rgIndices = NULL;
	SAFE_ALLOC(rgIndices, LONG, cDimensions);

	//Loop over all dimenstions and fill in "pre" info...
	for(iDim=0; iDim<cDimensions; iDim++)
	{
		//Fill in lower bound
		LONG lLowerBound = 0;
		SafeArrayGetLBound(pSafeArray, iDim+1, &lLowerBound);
		rgIndices[iDim] = lLowerBound;
		
		//Fill in outer dimension indicater
		strcat(pszBuffer, "[");
	}
	
	//Calculate the total number of inner items...
	//This is easy, all dimensions will have the same number "inner" items
	//IE:  rg[y][x] - all y arrays have x elements.
	//IE:  rg[z][y][x] - all z arrays, have y arrays, which have x elements
	ulInnerElements = pSafeArray->rgsabound[0].cElements;
	while(!bDone)
	{	
		//Dimension[0] always goes through a complete cycle every time
		//Just do this part of the ripple counter seperately...
		for(i=0; i<ulInnerElements; i++)
		{
			//Initialize Variant
			VariantInit(&Variant);
			V_VT(&Variant) = wType;
			
			//Obtain the Data from the SafeArray
			TESTC(hr = SafeArrayGetElement(pSafeArray, rgIndices, &V_I4(&Variant)));
			rgIndices[0]++;

			//Adjust next string placement
			ulStrLen = strlen(psz);
			psz = psz + ulStrLen;
			ulSizeLeft -= ulStrLen;

			//Convert VARIANT To String
			TESTC(hr = VariantToString(&Variant, psz, ulSizeLeft, CONV_NONE));
			
			//Array Seperator
			if(i<ulInnerElements-1)
				strcat(psz, ",");

			//Clear the Variant, (could be outofline memory...)
			XTEST(NULL, VariantClear(&Variant));
		}

		//Adjust the other Dimensions of the ripple counter
		for(iDim=0; iDim<cDimensions; iDim++)
		{
			LONG lUpperBound = 0;
			SafeArrayGetUBound(pSafeArray, iDim+1, &lUpperBound);
			
			//Increment this ripple if below max bound, and exit out (break)
			if(rgIndices[iDim] < lUpperBound)
			{
				rgIndices[iDim]++;

				//Need to add opening braces...
				for(ULONG j=iDim; j>0; j--)
					strcat(psz, "[");
				break;
			}
			//Otherwise reset this one and move onto the 
			//next Dimension (ie: Don't break...)
			else if(iDim != cDimensions-1)
			{
				LONG lLowerBound = 0;
				SafeArrayGetLBound(pSafeArray, iDim+1, &lLowerBound);
				rgIndices[iDim] = lLowerBound;
				strcat(psz, "]");
			}
			//If we have hit the last Dimension and its over the value
			//This means were done...
			else
			{
				bDone = TRUE;
			}
		}
	}

	//Display Right outer braces
	for(iDim=0; iDim<cDimensions; iDim++)
		strcat(pszBuffer, "]");

CLEANUP:
	XTEST(NULL, VariantClear(&Variant));
	SAFE_FREE(rgIndices);
	return hr;
}


/////////////////////////////////////////////////////////////////////////////
// HRESULT StringToSafeArray
//
/////////////////////////////////////////////////////////////////////////////
HRESULT StringToSafeArray(WCHAR* pwszBuffer, DBTYPE wType, SAFEARRAY** ppSafeArray)
{
	ASSERT(pwszBuffer);
	ASSERT(ppSafeArray);
	ASSERT(!(wType & VT_ARRAY));

	//This method is capable of handling N-Dimenstions of Data!!
	//We need to a String of N Dimensions and turn it into a SafeArray

	//For example:  We have a 3D data, where z-dim has 2 elements, y-Dim has 3 ele
	//and x-dim has 2 elements we end up with the following matrix of points
	
	//	(z, y, x)  => value	[[[
	//	(1, 1, 1)  => 1
	//	(1, 1, 2)  => 2
	//	(1, 2, 1)  => 3		][
	//	(1, 2, 2)  => 4
	//	(1, 3, 1)  => 5		][
	//	(1, 3, 2)  => 6
	//	(2, 1, 1)  => 7		]][[
	//	(2, 1, 2)  => 8
	//	(2, 2, 1)  => 9		][
	//	(2, 2, 2)  => A
	//	(2, 3, 1)  => B		][
	//	(2, 3, 2)  => C
	//						]]]

	//So we could be passed a string of:
	//	[ [[1,2][3,4][5,6]] [[7,8][9,A][B,C]] ]

	//The algorythm we are using is bascially based upon a "ripple" counter.
	//We keep a counter for each dimension.  So when CounterDim[0] hits the Upper
	//Limit, we increment CounterDim[1] and set CounterDim[0] back to LowerLimit.  
	//This continues until all have reached the upper limit together:
	//{CounterDim[n-1]==UpperLimt, ... Dim[0]==UpperLimit}

	//The braces are determined by rising/falling edges of the ripple counter.
	//Everytime a dimension restarts its value from Upper->Lower we see a "][".
	//So we have a pre and a post set of - "[[[" braces "]]]" for the number of dimensions.
	//You'll notices the set of braces in the above example on the rising/falling
	//edges of the ripple counter....

	HRESULT hr = S_OK;
	VARIANT Variant;
	VariantInit(&Variant);

	WCHAR wszBuffer[MAX_COL_SIZE];
	wszBuffer[0] = EOL;
	WCHAR* pwsz = pwszBuffer;

	ULONG i,iDim, ulInnerElements = 0;

	//Determine the Number of Dimensions...
	ULONG cDimensions = 0;
	while(pwsz[0]==L'[')
	{
		cDimensions++;
		pwsz++;
		wcscat(wszBuffer, L"]");
	}

	//Find the End of the Data (where everever "]...") is...
	WCHAR* pwszNext = pwsz;
	WCHAR* pwszEnd = pwsz;
	WCHAR* pwszCurEnd = wcsstr(pwsz, L"]");
	WCHAR* pwszEndString = wcsstr(pwsz, wszBuffer);

	//No-op
	if(cDimensions < 1)
		return E_FAIL;

	//Create SafeArray
	SAFEARRAY* pSafeArray = NULL;
	*ppSafeArray = NULL;

	LONG* rgIndices = NULL;
	SAFEARRAYBOUND* rgSafeArrayBounds = NULL;

	//Indices array...
	SAFE_ALLOC(rgIndices, LONG, cDimensions);
	memset(rgIndices, 0, cDimensions*sizeof(LONG));

	//SafeArrayBounds array...
	SAFE_ALLOC(rgSafeArrayBounds, SAFEARRAYBOUND, cDimensions);
	memset(rgSafeArrayBounds, 0, cDimensions*sizeof(SAFEARRAYBOUND));

	//Need to find out how many elements are in Dim[0]
	while(pwszNext && pwszNext < pwszCurEnd)
	{
		ulInnerElements++;
		rgIndices[0]++;
		pwszNext = wcsstr(pwszNext, L",");
		if(pwszNext)
			pwszNext++;
	}

	//Now from the [] notation find out how many elements in the other dimenstions
	//	[ [[1,2][3,4][5,6]] [[7,8][9,A][B,C]] ]
	
	//The algorythm we will use is:
	//Everytime we see a "]" we need to increment the next Dimension elements
	//Everytime we see a "[" we need to reset the previous Dimension elements
	
	//TODO
	//Currently this algorythm only handles 1 dimension.
	//No reason it couldn't use the above method and work for more, just on
	//a time constraint, and the only provider we have supports max 1 dim...

	ASSERT(cDimensions == 1);
	
	//Create the SafeArrayBounds
	for(iDim=0; iDim<cDimensions; iDim++)
	{
		rgSafeArrayBounds[iDim].lLbound = 0;
		rgSafeArrayBounds[iDim].cElements = rgIndices[0];
	}
	
	//Now actually create the SafeArray
	pSafeArray = SafeArrayCreate(wType, cDimensions, rgSafeArrayBounds);
	ASSERT(pSafeArray);
	
	pwszCurEnd = wcsstr(pwsz, L"]");
	rgIndices[0] = 0;
	pwszNext = pwsz;
	for(i=0; i<ulInnerElements; i++)
	{
		//Obtain the start and end of the value
		pwszEnd = wcsstr(pwszNext, L",");
		if(pwszEnd==NULL || pwszEnd>pwszCurEnd)
			pwszEnd = pwszCurEnd;

		//Setup rgIndicaes
		rgIndices[0] = i;

		//Convert Value to a Variant
		wcsncpy(wszBuffer, pwszNext, pwszEnd-pwszNext);
		wszBuffer[pwszEnd-pwszNext] = wEOL;
		XTESTC(NULL, hr = StringToVariant(wszBuffer, wType, &Variant, CONV_NONE));

		//Add this Value to the SafeArray
		TESTC(hr = SafeArrayPutElement(pSafeArray, rgIndices, &V_I4(&Variant)));
		XTEST(NULL, VariantClear(&Variant));

		//Incement to next value...
		pwszNext = wcsstr(pwszNext, L",");
		if(pwszNext)
			pwszNext += 1;
	}

	//Everything complete successfully...
	*ppSafeArray = pSafeArray;

CLEANUP:
	XTEST(NULL, VariantClear(&Variant));
	SAFE_FREE(rgIndices);
	SAFE_FREE(rgSafeArrayBounds);
	return hr;
}


/////////////////////////////////////////////////////////////////////////////
// CHAR* strDuplicate
//
/////////////////////////////////////////////////////////////////////////////
CHAR* strDuplicate(CHAR* psz)
{
	//no-op case
	if(!psz)
		return NULL;
	
	ULONG cLen	= strlen(psz);

	//Allocate space for the string
	CHAR* pszBuffer = NULL;
	SAFE_ALLOC(pszBuffer, CHAR, cLen+1);

	//Now copy the string
	strcpy(pszBuffer, psz);

CLEANUP:
	return pszBuffer;
}



/////////////////////////////////////////////////////////////////////////////
// WCHAR* wcsDuplicate
//
/////////////////////////////////////////////////////////////////////////////
WCHAR* wcsDuplicate(WCHAR* pwsz)
{
	//no-op case
	if(!pwsz)
		return NULL;
	
	ULONG cLen	= wcslen(pwsz);

	//Allocate space for the string
	WCHAR* pwszBuffer = NULL;
	SAFE_ALLOC(pwszBuffer, WCHAR, cLen+1);

	//Now copy the string
	wcscpy(pwszBuffer, pwsz);

CLEANUP:
	return pwszBuffer;
}



/////////////////////////////////////////////////////////////////////////////
// HRESULT DBIDToString
//
/////////////////////////////////////////////////////////////////////////////
HRESULT DBIDToString(DBID* pDBID, WCHAR* pwsz, ULONG ulMaxLen)
{
	ASSERT(pwsz);
	WCHAR pwszBuffer[MAX_NAME_LEN];

	//No-op
	if(pDBID == NULL)
	{
		wcscpy(pwsz, L"NULL");
		return S_OK;
	}

	//ColumnID (SubItem)  DBID
	switch(pDBID->eKind)
	{
		case DBKIND_NAME:
			swprintf(pwsz, L"{\"%s\"}", pDBID->uName.pwszName);
			break;
		
		case DBKIND_PROPID:
			swprintf(pwsz, L"{%d}", pDBID->uName.ulPropid);
			break;
		
		case DBKIND_GUID:
			StringFromGUID2(pDBID->uGuid.guid, pwsz, ulMaxLen);
			break;

		case DBKIND_GUID_NAME:
			StringFromGUID2(pDBID->uGuid.guid, pwszBuffer, ulMaxLen);
			swprintf(pwsz, L"{%s,\"%s\"}", pwszBuffer, pDBID->uName.pwszName);
			break;
		
		case DBKIND_GUID_PROPID:
			StringFromGUID2(pDBID->uGuid.guid, pwszBuffer, ulMaxLen);
			swprintf(pwsz, L"{%s,%d}", pwszBuffer, pDBID->uName.ulPropid);
			break;
		
		case DBKIND_PGUID_NAME:
			StringFromGUID2(*pDBID->uGuid.pguid, pwszBuffer, ulMaxLen);
			swprintf(pwsz, L"{&%s,\"%s\"}", pwszBuffer, pDBID->uName.pwszName);
			break;
		
		case DBKIND_PGUID_PROPID:
			StringFromGUID2(*pDBID->uGuid.pguid, pwszBuffer, ulMaxLen);
			swprintf(pwsz, L"{&%s,%d}", pwszBuffer, pDBID->uName.ulPropid);
			break;
		
		default:
			ASSERT(!"Unhandled Type!");
			break;
	};

	return S_OK;
}


/////////////////////////////////////////////////////////////////////////////
// BOOL IsVariableType
//
/////////////////////////////////////////////////////////////////////////////
BOOL IsVariableType(DBTYPE wType)
{
	//According to OLEDB Spec Appendix A (Variable-Length Data Types)
	switch(wType) 
	{
		case DBTYPE_STR:
		case DBTYPE_WSTR:
		case DBTYPE_BYTES:
			return TRUE;
	}
	return FALSE;
}


/////////////////////////////////////////////////////////////////////////////
// BOOL IsFixedType
//
/////////////////////////////////////////////////////////////////////////////
BOOL IsFixedType(DBTYPE wType)
{
	return !IsVariableType(wType);
}

/////////////////////////////////////////////////////////////////////////////
// BOOL IsNumericType
//
/////////////////////////////////////////////////////////////////////////////
BOOL IsNumericType(DBTYPE wType)
{
	//According to OLEDB Spec Appendix A (Numeric Data Types)
	switch(wType) 
	{
		case DBTYPE_I1:
		case DBTYPE_I2:
		case DBTYPE_I4:
		case DBTYPE_I8:
		case DBTYPE_UI1:
		case DBTYPE_UI2:
		case DBTYPE_UI4:
		case DBTYPE_UI8:
		case DBTYPE_R4:
		case DBTYPE_R8:
		case DBTYPE_CY:
		case DBTYPE_DECIMAL:
		case DBTYPE_NUMERIC:
			return TRUE;
	}
	return FALSE;
}


/////////////////////////////////////////////////////////////////////////////
// HRESULT GetDBTypeMaxSize
//
/////////////////////////////////////////////////////////////////////////////
HRESULT GetDBTypeMaxSize(DBTYPE wType, ULONG* pulMaxSize, BYTE* pbPrecision, BYTE* pbScale)
{
	HRESULT hr = S_OK;
	ULONG	ulMaxSize = 0;
	BYTE	bPrecision = 0;
	BYTE	bScale = 0;

	//Values taken from OLE DB Spec, Appendix A
	//In some situations we need to know ulMaxSize, Prec, Scale, as defaults or creating
	//an accessor before we actually have ColumnsInfo.  Useful info, but mainly used
	//only for defaults for dialogs...

	//Unhandled Modifiers
	if ((wType & DBTYPE_RESERVED) ||
		(wType & DBTYPE_ARRAY)	 ||
		(wType & DBTYPE_VECTOR))
	{
		hr = E_FAIL;
		goto CLEANUP;
	}

	if(wType & DBTYPE_BYREF)
	{
		ulMaxSize = sizeof(void*);
		goto CLEANUP;
	}
	
	switch(wType)
	{
		case DBTYPE_EMPTY:
		case DBTYPE_NULL:
			break;
		
		case DBTYPE_STR:
		case DBTYPE_WSTR:
		case DBTYPE_BYTES:
		case DBTYPE_BSTR:
			ulMaxSize = MAX_COL_SIZE;
			break;

		case DBTYPE_I1:
		case DBTYPE_UI1:
			ulMaxSize = 1;
			bPrecision = 3;
			break;

		case DBTYPE_I2:
		case DBTYPE_UI2:
			ulMaxSize = 2;
			bPrecision = 5;
			break;

		case DBTYPE_I4:
		case DBTYPE_UI4:
			ulMaxSize = 4;
			bPrecision = 10;
			break;

		case DBTYPE_I8:
			ulMaxSize = 8;
			bPrecision = 19;
			break;

		case DBTYPE_UI8:
			ulMaxSize = 8;
			bPrecision = 20;
			break;

		case DBTYPE_R4:
			ulMaxSize = sizeof(float);
			bPrecision = 7;
			break;

		case DBTYPE_R8:
			ulMaxSize = sizeof(double);
			bPrecision = 16;
			break;

		case DBTYPE_CY:
			ulMaxSize = 8;
			bPrecision = 19;
			break;

		case DBTYPE_NUMERIC:
			ulMaxSize = sizeof(DB_NUMERIC);
			bPrecision = 38;
			break;

		case DBTYPE_DATE:
			ulMaxSize = sizeof(double);
			break;

		case DBTYPE_BOOL:
			ulMaxSize = 2;
			break;

		case DBTYPE_VARIANT:
			ulMaxSize = sizeof(VARIANT);
			break;

		case DBTYPE_IDISPATCH:
			ulMaxSize = sizeof(IDispatch*);
			break;

		case DBTYPE_IUNKNOWN:
			ulMaxSize = sizeof(IUnknown*);
			break;

		case DBTYPE_GUID:
			ulMaxSize = sizeof(GUID);
			break;

		case DBTYPE_ERROR:
			ulMaxSize = sizeof(SCODE);
			break;

		case DBTYPE_DBDATE:
			ulMaxSize = sizeof(DBDATE);
			break;

		case DBTYPE_DBTIME:
			ulMaxSize = sizeof(DBTIME);
			break;

		case DBTYPE_DBTIMESTAMP:
			ulMaxSize = sizeof(DBTIMESTAMP);
			break;

		case DBTYPE_DECIMAL:
			ulMaxSize = sizeof(DECIMAL);
			bPrecision = 28;
			break;
	}

CLEANUP:
	if(pulMaxSize)
		*pulMaxSize = ulMaxSize;
	if(pbPrecision)
		*pbPrecision = bPrecision;
	if(pbScale)
		*pbScale = bScale;
	return hr;
}

/////////////////////////////////////////////////////////////////////////////
// WCHAR* GetDBTypeName
//
/////////////////////////////////////////////////////////////////////////////
WCHAR* GetDBTypeName(DBTYPE wType)
{
	return GetMapName(wType, g_cDBTypes, g_rgDBTypes);
}


/////////////////////////////////////////////////////////////////////////////
// DBTYPE GetDBType
//
/////////////////////////////////////////////////////////////////////////////
DBTYPE GetDBType(WCHAR* pwsz)
{
	return (DBTYPE)GetMapName(pwsz, g_cDBTypes, g_rgDBTypes);
}


/////////////////////////////////////////////////////////////////////////////
// CHAR* GetVariantTypeName
//
/////////////////////////////////////////////////////////////////////////////
CHAR* GetVariantTypeName(VARTYPE vt)
{
	return GetMapName(vt, g_cVariantTypes, g_rgVariantTypes);
}


/////////////////////////////////////////////////////////////////////////////
// CHAR* GetVariantTypeName
//
/////////////////////////////////////////////////////////////////////////////
DBTYPE GetVariantType(CHAR* psz)
{
	return (DBTYPE)GetMapName(psz, g_cVariantTypes, g_rgVariantTypes);
}


///////////////////////////////////////////////////////////////
// BOOL FreeBindingData
//
///////////////////////////////////////////////////////////////
BOOL FreeBindingData(ULONG cBindings, DBBINDING* rgBindings, void* pData)
{
	ASSERT(pData);
	
	//Need to walk the array and free any other alloc memory
	for(ULONG i=0; i<cBindings; i++)
	{
		//Free any "out-of-line" memory
		//VARIANT
		if(rgBindings[i].wType == DBTYPE_VARIANT)
		{
			VARIANT* pVariant = (VARIANT*)&BINDING_VALUE(rgBindings[i], pData);
			FreeVariants(1, pVariant);
		}
	}

	return TRUE;
}


///////////////////////////////////////////////////////////////
// BOOL FreeBindings
//
///////////////////////////////////////////////////////////////
BOOL FreeBindings(ULONG* pcBindings, DBBINDING** prgBindings)
{
	ASSERT(pcBindings);
	ASSERT(prgBindings);

	//Need to walk the array and free any other alloc memory
	for(ULONG i=0; i<*pcBindings; i++)
	{
		//Free any pObjects
		ASSERT(*prgBindings);
		SAFE_FREE((*prgBindings)[i].pObject);
	}

	//Now we can free the outer struct
	*pcBindings = 0;
	SAFE_FREE(*prgBindings);
	return TRUE;
}


//////////////////////////////////////////////////////////////////
// void SyncSibling
//
//////////////////////////////////////////////////////////////////
void SyncSibling(HWND hToWnd, HWND hFromWnd)
{
	ASSERT(hToWnd && hFromWnd);

	//Make both windows synched, 
	//Get the current selection from the Source
	LONG iItem = SendMessage(hFromWnd, LVM_GETNEXTITEM, (WPARAM)-1, (LPARAM)LVNI_SELECTED);
	
	//Tell the Target to select the same selection
	if(iItem != LVM_ERR)
	{
		//Get the current selection from the Target and Unselect it
		LONG iOldItem = SendMessage(hToWnd, LVM_GETNEXTITEM, (WPARAM)-1, (LPARAM)LVNI_SELECTED);
		if(iItem != iOldItem)
		{
			//Unselect previous one
			LV_SetItemState(hToWnd, iOldItem, 0, 0, LVIS_SELECTED);

			//Select the new one
			LV_SetItemState(hToWnd, iItem, 0, LVIS_SELECTED, LVNI_SELECTED);

			//Ensure that it is visible
			SendMessage(hToWnd, LVM_ENSUREVISIBLE, (WPARAM)iItem, (LPARAM)FALSE);
		}
	}
}                



/////////////////////////////////////////////////////////////////////
// Exception Handlers
//
/////////////////////////////////////////////////////////////////////
HRESULT HandleException(HWND hWnd, EXCEPTION_SOURCE eSource, WCHAR* pwszFile, ULONG ulLine)
{
	//Popup a MessageBox
	LONG dwSelection = wMessageBox(
						NULL, 
						MB_TASKMODAL | MB_ICONSTOP | MB_OKCANCEL | MB_DEFBUTTON1,
						wsz_EXCEPTION, 
						L"%s\n"
						L"File '%s', Line '%lu'\n\n"
						L"Do you wish to Continue?\n"
						L"(Press 'OK' to ignore the Exception.  Press 'Cancel' to debug.)",
						
						(eSource == EXCEPTION_PROVIDER) ? 
							L"The OLE DB Provider your using threw an Exception!" : 
							L"RowsetViewer threw an Exception!",
						pwszFile,
						ulLine
						);

	switch(dwSelection)
	{
		case IDOK:
			return 0;
		case IDCANCEL:
			return 1;
		default:
			ASSERT(!L"Unhandled Choice");
	}

	return 0;
}


//////////////////////////////////////////////////////////////////
// int InternalAssert
//
//////////////////////////////////////////////////////////////////
int InternalAssert(					// 1 to break, 0 to skip.
    char*	pszExp,					// The expression causing assert
	char*	pszFile,				// The file name
	UINT	iLine					// Line number of assert
	)
{
	//Popup a MessageBox
	LONG dwSelection = wMessageBox(
						NULL, 
						MB_TASKMODAL | MB_ICONSTOP | MB_OKCANCEL | MB_DEFBUTTON1,
						wsz_ERROR, 
						L"Assertion Error!\nFile '%S', Line '%lu'\n"
						L"Expression '%S'\n\n"
						L"Do you wish to Continue?\n"
						L"(Press 'OK' to ignore the assertion.  Press 'Cancel' to debug.)",
						
						pszFile,
						iLine,
						pszExp
						);

	switch(dwSelection)
	{
		case IDOK:
			return 0;
		case IDCANCEL:
			return 1;
		default:
			ASSERT(!L"Unhandled Choice");
	}

	return 0;
}


//////////////////////////////////////////////////////////////////
// void InternalTrace
//
//////////////////////////////////////////////////////////////////
void InternalTrace(WCHAR*	pwszFmt, ...)
{
	va_list		marker;
	WCHAR		wszBuffer[MAX_QUERY_LEN];
	CHAR		szBuffer[MAX_QUERY_LEN];

	// Use format and arguements as input
	//This version will not overwrite the stack, since it only copies
	//upto the max size of the array
	va_start(marker, pwszFmt);
	_vsnwprintf(wszBuffer, MAX_QUERY_LEN, pwszFmt, marker);
	va_end(marker);

	//Make sure there is a NULL Terminator, vsnwprintf will not copy
	//the terminator if length==MAX_QUERY_LEN
	wszBuffer[MAX_QUERY_LEN-1] = wEOL;
	
	//Convert to MBCS
	ConvertToMBCS(wszBuffer, szBuffer, MAX_QUERY_LEN);
	
	//Output to the DebugWindow
	OutputDebugString(szBuffer);
}


//////////////////////////////////////////////////////////////////
// void InternalTrace
//
//////////////////////////////////////////////////////////////////
void InternalTrace(CHAR*	pszFmt, ...)
{
	va_list		marker;
	CHAR		szBuffer[MAX_QUERY_LEN];

	// Use format and arguements as input
	//This version will not overwrite the stack, since it only copies
	//upto the max size of the array
	va_start(marker, pszFmt);
	_vsnprintf(szBuffer, MAX_QUERY_LEN, pszFmt, marker);
	va_end(marker);

	//Make sure there is a NULL Terminator, vsnwprintf will not copy
	//the terminator if length==MAX_QUERY_LEN
	szBuffer[MAX_QUERY_LEN-1] = EOL;
	
	OutputDebugStringA(szBuffer);
}


//////////////////////////////////////////////////////////////////
// void Busy
//
//////////////////////////////////////////////////////////////////
void Busy(BOOL bValue)
{
	static HCURSOR	hWaitCursor = LoadCursor(NULL, IDC_WAIT);

	if(bValue) 
		SetCursor(hWaitCursor);
	else 
		SetCursor(NULL);
}


//////////////////////////////////////////////////////////////////
// void OutOfMemory
//
//////////////////////////////////////////////////////////////////
void OutOfMemory(HWND hWnd)
{
	//Unicode version is supported on Win95/WinNT
	MessageBoxW(hWnd, L"Out of memory", wsz_ERROR, MB_TASKMODAL | MB_OK);
}


//////////////////////////////////////////////////////////////////
// BOOL CenterDialog
//
//////////////////////////////////////////////////////////////////
BOOL CenterDialog(HWND hdlg)
{
	RECT  rcParent;                         // Parent window client rect
	RECT  rcDlg;                            // Dialog window rect
	int   nLeft, nTop;                      // Top-left coordinates
	int   cWidth, cHeight;                  // Width and height
	HWND	hwnd;

	// Get frame window client rect in screen coordinates
	hwnd = GetParent(hdlg);
	if(hwnd == NULL || hwnd == GetDesktopWindow()) 
	{
		rcParent.top = rcParent.left = 0;
		rcParent.right = GetSystemMetrics(SM_CXFULLSCREEN);
		rcParent.bottom = GetSystemMetrics(SM_CYFULLSCREEN);
	}
	else 
		GetWindowRect(hwnd, &rcParent);

	// Determine the top-left point for the dialog to be centered
	GetWindowRect(hdlg, &rcDlg);
	cWidth  = rcDlg.right  - rcDlg.left;
	cHeight = rcDlg.bottom - rcDlg.top;
	nLeft   = rcParent.left + 
            (((rcParent.right  - rcParent.left) - cWidth ) / 2);
	nTop    = rcParent.top  +
            (((rcParent.bottom - rcParent.top ) - cHeight) / 2);
	if (nLeft < 0) nLeft = 0;
	if (nTop  < 0) nTop  = 0;

	// Place the dialog
	return MoveWindow(hdlg, nLeft, nTop, cWidth, cHeight, TRUE);
}


//////////////////////////////////////////////////////////////////
// BOOL MoveWindow
//
//////////////////////////////////////////////////////////////////
BOOL MoveWindow(HWND hWnd, ULONG x, ULONG y)
{
	RECT  rcParent;                         // Parent window client rect
	RECT  rcDlg;                            // Dialog window rect
	int   nLeft, nTop;                      // Top-left coordinates
	int   cWidth, cHeight;                  // Width and height
	HWND  hWndParent;

	// Get frame window client rect in screen coordinates
	hWndParent = GetParent(hWnd);
	if(hWndParent == NULL || hWndParent == GetDesktopWindow()) 
	{
		rcParent.top = rcParent.left = 0;
		rcParent.right = GetSystemMetrics(SM_CXFULLSCREEN);
		rcParent.bottom = GetSystemMetrics(SM_CYFULLSCREEN);
	}
	else 
		GetWindowRect(hWndParent, &rcParent);

	// Determine the top-left point for the dialog to be centered
	GetWindowRect(hWnd, &rcDlg);
	cWidth  = rcDlg.right  - rcDlg.left;
	cHeight = rcDlg.bottom - rcDlg.top;
	nLeft   = rcParent.left + x;
	nTop    = rcParent.top + y;

	// Place the dialog
	return MoveWindow(hWnd, nLeft, nTop, cWidth, cHeight, TRUE);
}

//////////////////////////////////////////////////////////////////
// BOOL GetWindowSize
//
//////////////////////////////////////////////////////////////////
SIZE GetWindowSize(HWND hWnd)
{
	RECT rect;
	SIZE size;

	//Obtain window cordinates.
	GetWindowRect(hWnd, &rect);
	
	//Fillin SIZE struct
	size.cx = rect.right - rect.left;
	size.cy = rect.bottom - rect.top;
	return size;
}

//////////////////////////////////////////////////////////////////
// ULONG wMessageBox
//
//////////////////////////////////////////////////////////////////
INT wMessageBox(
	HWND hwnd,							// Parent window for message display
	UINT uiStyle,						// Style of message box
	WCHAR* pwszTitle,					// Title for message
	WCHAR* pwszFmt,						// Format string
	...									// Substitution parameters
	)
{
	va_list		marker;
	WCHAR		wszBuffer[MAX_QUERY_LEN];

	// Use format and arguements as input
	//This version will not overwrite the stack, since it only copies
	//upto the max size of the array
	va_start(marker, pwszFmt);
	_vsnwprintf(wszBuffer, MAX_QUERY_LEN, pwszFmt, marker);
	va_end(marker);
   
	//Make sure there is a NULL Terminator, vsnwprintf will not copy
	//the terminator if length==MAX_QUERY_LEN
	wszBuffer[MAX_QUERY_LEN-1] = wEOL;

	//Unicode version is supported on both Win95 / WinNT do need to convert
	return MessageBoxW(hwnd, wszBuffer, pwszTitle, uiStyle);
}


//////////////////////////////////////////////////////////////////
// INT MessageBox
//
//////////////////////////////////////////////////////////////////
INT MessageBox(
	HWND hwnd,							// Parent window for message display
	UINT uiStyle,						// Style of message box
	CHAR* pszTitle,						// Title for message
	CHAR* pszFmt,						// Format string
	...									// Substitution parameters
	)
{
	va_list		marker;
	CHAR		szBuffer[MAX_QUERY_LEN];

	// Use format and arguements as input
	//This version will not overwrite the stack, since it only copies
	//upto the max size of the array
	va_start(marker, pszFmt);
	_vsnprintf(szBuffer, MAX_QUERY_LEN, pszFmt, marker);
	va_end(marker);
   
	//Make sure there is a NULL Terminator, vsnwprintf will not copy
	//the terminator if length==MAX_QUERY_LEN
	szBuffer[MAX_QUERY_LEN-1] = EOL;

	//Delegate
	return MessageBoxA(hwnd, szBuffer, pszTitle, uiStyle);
}


//////////////////////////////////////////////////////////////////
// LRESULT wSendMessage
//
//////////////////////////////////////////////////////////////////
LRESULT wSendMessage(HWND hWnd, UINT Msg, WPARAM wParam, WCHAR* pwszBuffer)
{
	CHAR szBuffer[MAX_QUERY_LEN];						  
	szBuffer[0] = EOL;
	
	if(pwszBuffer && Msg != WM_GETTEXT && Msg != CB_GETLBTEXT)
	{
		//Convert to ANSI before sending, since we don't know if this was a GET/SET message
		ConvertToMBCS(pwszBuffer, szBuffer, MAX_QUERY_LEN);
	}


	//Send the message with an ANSI Buffer 
	LRESULT lResult = SendMessageA(hWnd, Msg, (WPARAM)wParam, (LPARAM)szBuffer);

	if(pwszBuffer && Msg == WM_GETTEXT || Msg == CB_GETLBTEXT)
	{
		//Now convert the result into the users WCHAR buffer
		ConvertToWCHAR(szBuffer, pwszBuffer, Msg == WM_GETTEXT ? wParam : MAX_QUERY_LEN);
	}
	return lResult;
}


//////////////////////////////////////////////////////////////////
// LRESULT wSendMessageFmt
//
//////////////////////////////////////////////////////////////////
LRESULT wSendMessageFmt(HWND hWnd, UINT Msg, WPARAM wParam, WCHAR* pwszFmt, ...)
{
	ASSERT(pwszFmt);
	ASSERT(wParam != WM_GETTEXT);
	
	va_list		marker;
	WCHAR		wszBuffer[MAX_QUERY_LEN];

	// Use format and arguements as input
	//This version will not overwrite the stack, since it only copies
	//upto the max size of the array
	va_start(marker, pwszFmt);
	_vsnwprintf(wszBuffer, MAX_QUERY_LEN, pwszFmt, marker);
	va_end(marker);

	//Make sure there is a NULL Terminator, vsnwprintf will not copy
	//the terminator if length==MAX_QUERY_LEN
	wszBuffer[MAX_QUERY_LEN-1] = wEOL;

	//Delegate
	return wSendMessage(hWnd, Msg, wParam, wszBuffer);
}


//////////////////////////////////////////////////////////////////
// LRESULT SendMessage
//
//////////////////////////////////////////////////////////////////
LRESULT SendMessageFmt(HWND hWnd, UINT Msg, WPARAM wParam, CHAR* pszFmt, ...)
{
	ASSERT(pszFmt);
	ASSERT(wParam != WM_GETTEXT);
	
	va_list		marker;
	CHAR		szBuffer[MAX_QUERY_LEN];

	// Use format and arguements as input
	//This version will not overwrite the stack, since it only copies
	//upto the max size of the array
	va_start(marker, pszFmt);
	_vsnprintf(szBuffer, MAX_QUERY_LEN, pszFmt, marker);
	va_end(marker);

	//Make sure there is a NULL Terminator, vsnwprintf will not copy
	//the terminator if length==MAX_QUERY_LEN
	szBuffer[MAX_QUERY_LEN-1] = EOL;
	
	//Delegate 
	return SendMessageA(hWnd, Msg, (WPARAM)wParam, (LPARAM)szBuffer);
}


//////////////////////////////////////////////////////////////////
// BOOL GetEditBoxValue
//
//////////////////////////////////////////////////////////////////
BOOL GetEditBoxValue(HWND hEditWnd, LONG lMin, LONG lMax, LONG* plValue, BOOL fAllowEmpty)
{
	ASSERT(hEditWnd);
	ASSERT(plValue);

	LONG	lValue = 0;
	WCHAR	wszBuffer[MAX_QUERY_LEN];
	wszBuffer[0] = wEOL;
	WCHAR*  pwszEnd = NULL;
	
	//Get the EditText
	wSendMessage(hEditWnd, WM_GETTEXT, MAX_QUERY_LEN, wszBuffer);
		
	if(wszBuffer[0] || !fAllowEmpty)
	{
		//Convert to LONG
		lValue = wcstol(wszBuffer, &pwszEnd, 0);
		if(!wszBuffer[0] || lValue<lMin || lValue>lMax || pwszEnd==NULL || pwszEnd[0]!=wEOL) 
		{
			wMessageBox(hEditWnd, MB_TASKMODAL | MB_ICONERROR | MB_OK,  wsz_ERROR, 
				wsz_INVALID_VALUE_, wszBuffer, lMin, lMax);
			SetFocus(hEditWnd);
			return FALSE;
		}

		*plValue = lValue;
	}
	return TRUE;
}


//////////////////////////////////////////////////////////////////
// LONG CB_GetSelectedText
//
//////////////////////////////////////////////////////////////////
LONG CB_GetSelectedText(HWND hWndCombo, CHAR* pszBuffer, ULONG ulMaxSize)
{
	ASSERT(pszBuffer);
	ASSERT(ulMaxSize);

	//Try to obtain the Current Selection
	LONG iSel = SendMessage(hWndCombo, CB_GETCURSEL, 0, 0);
	
	//This may fail, if the current selection is entered (DropDown instead of DropList)
	if(iSel == CB_ERR)
	{
		SendMessage(hWndCombo, WM_GETTEXT, ulMaxSize, (LPARAM)pszBuffer);
	}
	else
	{
		//Should be limiting the text if this ASSERT is hit!
		//Length does not include the NULL Terminator
		ULONG ulLength = SendMessage(hWndCombo, CB_GETLBTEXTLEN, iSel, 0);
		ASSERT(ulLength < ulMaxSize);
		
		//Obtain the text...
		if(ulLength < ulMaxSize)
			SendMessage(hWndCombo, CB_GETLBTEXT, iSel, (LPARAM)pszBuffer);
	}

	return iSel;
}


//////////////////////////////////////////////////////////////////
// LONG CB_GetSelectedText
//
//////////////////////////////////////////////////////////////////
LONG CB_GetSelectedText(HWND hWndCombo, WCHAR* pwszBuffer, ULONG ulMaxSize)
{
	ASSERT(pwszBuffer);
	ASSERT(ulMaxSize);

	//Try to obtain the Current Selection
	LONG iSel = SendMessage(hWndCombo, CB_GETCURSEL, 0, 0);
	
	//This may fail, if the current selection is entered (DropDown instead of DropList)
	if(iSel == CB_ERR)
	{
		wSendMessage(hWndCombo, WM_GETTEXT, ulMaxSize, pwszBuffer);
	}
	else
	{
		//Should be limiting the text if this ASSERT is hit!
		//Length does not include the NULL Terminator
		ULONG ulLength = SendMessage(hWndCombo, CB_GETLBTEXTLEN, iSel, 0);
		ASSERT(ulLength < ulMaxSize);
		
		//Obtain the text...
		if(ulLength < ulMaxSize)
			wSendMessage(hWndCombo, CB_GETLBTEXT, iSel, pwszBuffer);
	}

	return iSel;
}


//////////////////////////////////////////////////////////////////
// LONG CB_SelectText
//
//////////////////////////////////////////////////////////////////
LONG CB_SelectText(HWND hWndCombo, WCHAR* pwszBuffer, BOOL fAddItem)
{
	ASSERT(pwszBuffer);
	
	//Try to find the Indicated Text
	LONG iSel = wSendMessage(hWndCombo, CB_FINDSTRINGEXACT, -1, pwszBuffer);
	
	//If not found, just add it to the list (if desired)
	if(iSel == CB_ERR)
	{
		if(fAddItem)
		{
			iSel = wSendMessage(hWndCombo, CB_ADDSTRING, 0, pwszBuffer);
		}
		else
		{
			SendMessage(hWndCombo, CB_SETCURSEL, iSel, 0);
			wSendMessage(hWndCombo, WM_SETTEXT, 0, pwszBuffer);
		}
	}
	
	if(iSel != CB_ERR)
		iSel = SendMessage(hWndCombo, CB_SETCURSEL, iSel, 0);

	return iSel;
}

					
//////////////////////////////////////////////////////////////////
// LONG CB_SelectText
//
//////////////////////////////////////////////////////////////////
LONG CB_SelectText(HWND hWndCombo, CHAR* pszBuffer, BOOL fAddItem)
{
	ASSERT(pszBuffer);
	
	//Try to find the Indicated Text
	LONG iSel = SendMessage(hWndCombo, CB_FINDSTRINGEXACT, -1, (LPARAM)pszBuffer);
	
	//If not found, just add it to the list (if desired)
	if(iSel == CB_ERR)
	{
		if(fAddItem)
		{
			iSel = SendMessage(hWndCombo, CB_ADDSTRING, 0, (LPARAM)pszBuffer);
		}
		else
		{
			SendMessage(hWndCombo, CB_SETCURSEL, iSel, 0);
			SendMessage(hWndCombo, WM_SETTEXT, 0, (LPARAM)pszBuffer);
		}
	}

	if(iSel != CB_ERR)
		iSel = SendMessage(hWndCombo, CB_SETCURSEL, iSel, 0);

	return iSel;
}


////////////////////////////////////////////////////////////////
// CB_SelectItemValue
//
/////////////////////////////////////////////////////////////////
LONG CB_SelectItemValue(HWND hWndCombo, LONG lParam)
{
	//Loop through all Combo Item Values and Select specified one...
	LONG iCount = SendMessage(hWndCombo, CB_GETCOUNT, 0, 0);
	for(LONG i=0; i<iCount; i++)
	{
		if(lParam == SendMessage(hWndCombo, CB_GETITEMDATA, i, 0))
		{
			SendMessage(hWndCombo, CB_SETCURSEL, i, 0);
			return i;			
		}
	}

	return CB_ERR;
}



//////////////////////////////////////////////////////////////////
// BOOL LV_InsertColumn
//
//////////////////////////////////////////////////////////////////
LONG LV_InsertColumn(HWND hWnd, LONG iColumn, CHAR* szName, LONG iImage)
{
	ULONG dwMask = LVCF_TEXT | LVCF_FMT | LVCF_SUBITEM;
	INT dwFmt = LVCFMT_LEFT;
	if(iImage != IMAGE_NONE)
	{
		dwMask |= LVCF_IMAGE;
		dwFmt |= LVCFMT_IMAGE;
	}
	
	//Setup LV_COLUMNINFO
	LV_COLUMN lvColumnHeader = { dwMask, dwFmt, 0, szName, 0, 0, iImage, 0};
	
	//LVM_INSERTCOLUMN
	return SendMessage(hWnd, LVM_INSERTCOLUMN, (WPARAM)iColumn, (LPARAM)&lvColumnHeader);
}


//////////////////////////////////////////////////////////////////
// BOOL LV_InsertItem
//
//////////////////////////////////////////////////////////////////
LONG LV_InsertItem(HWND hWnd, LONG iItem, LONG iSubItem, CHAR* szName, LONG iParam, LONG iImage)
{
	//Calculate the Mask/flags
	ULONG dwMask = 0;
	if(szName)
		dwMask |= LVIF_TEXT;
	if(iImage != IMAGE_NONE)
		dwMask |= LVIF_IMAGE;
	if(iSubItem == 0)
		dwMask |= LVIF_PARAM;
	
	//LVM_INSERTITEM
	if(iSubItem==0)
	{
		LV_ITEM lvItem = { dwMask, iItem, iSubItem, 0, 0, szName, 0, iImage, iParam, 0};
		return SendMessage(hWnd, LVM_INSERTITEM, 0, (LPARAM)&lvItem);
	}
	//LVM_SETITEM
	else
	{
		LV_ITEM lvItem = { dwMask, iItem, iSubItem, 0, 0, szName, 0, iImage, iParam, 0};
		return SendMessage(hWnd, LVM_SETITEM, 0, (LPARAM)&lvItem);
	}
}


//////////////////////////////////////////////////////////////////
// BOOL LV_SetItemText
//
//////////////////////////////////////////////////////////////////
LONG LV_SetItemText(HWND hWnd, LONG iItem, LONG iSubItem, CHAR* szName)
{
	//LVM_SETITEM
	LV_ITEM lvItem = { LVIF_TEXT, iItem, iSubItem, 0, 0, szName, 0, 0, 0, 0};
	return SendMessage(hWnd, LVM_SETITEMTEXT, (WPARAM)iItem, (LPARAM)&lvItem);
}

//////////////////////////////////////////////////////////////////
// BOOL LV_SetItemState
//
//////////////////////////////////////////////////////////////////
LONG LV_SetItemState(HWND hWnd, LONG iItem, LONG iSubItem, LONG lState, LONG lStateMask)
{
	//LVM_SETITEM
	LV_ITEM lvItem = { LVIF_STATE, iItem, iSubItem, lState, lStateMask, NULL, 0, 0, 0, 0};
	return SendMessage(hWnd, LVM_SETITEMSTATE, (WPARAM)iItem, (LPARAM)&lvItem);
}

//////////////////////////////////////////////////////////////////
// BOOL LV_SetItemImage
//
//////////////////////////////////////////////////////////////////
LONG LV_SetItemImage(HWND hWnd, LONG iItem, LONG iSubItem, LONG iImage)
{
	//LVM_SETITEM (With IMAGE mask)
	LV_ITEM lvItem = { LVIF_IMAGE, iItem, iSubItem, 0, 0, NULL, 0, iImage, 0, 0};
	return SendMessage(hWnd, LVM_SETITEM, 0, (LPARAM)&lvItem);
}

//////////////////////////////////////////////////////////////////
// BOOL LV_SetItemParam
//
//////////////////////////////////////////////////////////////////
LONG LV_SetItemParam(HWND hWnd, LONG iItem, LONG iSubItem, LONG lParam)
{
	//LVM_SETITEM (With IMAGE mask)
	LV_ITEM lvItem = { LVIF_PARAM, iItem, iSubItem, 0, 0, NULL, 0, 0, lParam, 0};
	return SendMessage(hWnd, LVM_SETITEM, 0, (LPARAM)&lvItem);
}

//////////////////////////////////////////////////////////////////
// BOOL LV_GetItemText
//
//////////////////////////////////////////////////////////////////
LONG LV_GetItemText(HWND hWnd, LONG iItem, LONG iSubItem, CHAR* szName, ULONG ulMaxSize)
{
	ASSERT(szName);
	szName[0] = EOL;
	
	//LVM_GETITEMTEXT
	LV_ITEM lvItem = { LVIF_TEXT, iItem, iSubItem, 0, 0, szName, ulMaxSize, 0, 0, 0};
	return SendMessage(hWnd, LVM_GETITEMTEXT, (WPARAM)iItem, (LPARAM)&lvItem);
}

//////////////////////////////////////////////////////////////////
// BOOL LV_GetItemState
//
//////////////////////////////////////////////////////////////////
LONG LV_GetItemState(HWND hWnd, LONG iItem, LONG iMask)
{
	//LVM_GETITEMSTATE
	return 	SendMessage(hWnd, LVM_GETITEMSTATE, (WPARAM)iItem, (LPARAM)iMask);
}

//////////////////////////////////////////////////////////////////
// BOOL LV_GetItemImage
//
//////////////////////////////////////////////////////////////////
LONG LV_GetItemImage(HWND hWnd, LONG iItem, LONG iSubItem)
{
	//LVM_GETITEM
	LV_ITEM lvItem = { LVIF_IMAGE, iItem, iSubItem, 0, 0, 0, 0, 0, 0, 0};
	SendMessage(hWnd, LVM_GETITEM, 0, (LPARAM)&lvItem);
	return lvItem.iImage;
}

//////////////////////////////////////////////////////////////////
// LONG LV_GetItemParam
//
//////////////////////////////////////////////////////////////////
LONG LV_GetItemParam(HWND hWnd, LONG iItem, LONG iSubItem)
{
	//LVM_GETITEM
	LV_ITEM lvItem = { LVIF_PARAM, iItem, iSubItem, 0, 0, 0, 0, 0, 0, 0};
	SendMessage(hWnd, LVM_GETITEM, 0, (LPARAM)&lvItem);
	return lvItem.lParam;
}


//////////////////////////////////////////////////////////////////
// LONG LV_GetSelItems
//
//////////////////////////////////////////////////////////////////
LONG LV_GetSelItems(HWND hWnd, ULONG* pcItems, LONG** prgSelItems, LONG** prgSelParams)
{
	//Get the total Selected Items
	LONG i,iSelItem =0;
	LONG cItems = SendMessage(hWnd, LVM_GETSELECTEDCOUNT, 0, 0);

	//Alloc Output Array
	if(prgSelItems)
		SAFE_ALLOC(*prgSelItems, LONG, cItems);
	if(prgSelParams)
		SAFE_ALLOC(*prgSelParams, LONG, cItems);

	//Find all params of Selected Items
	iSelItem = SendMessage(hWnd, LVM_GETNEXTITEM, (WPARAM)-1, (LPARAM)LVNI_SELECTED);
	for(i=0; i<cItems; i++)
	{
		if(prgSelItems)
			(*prgSelItems)[i] = iSelItem;
		if(prgSelParams)
			(*prgSelParams)[i] = LV_GetItemParam(hWnd, iSelItem, 0);
		iSelItem = SendMessage(hWnd, LVM_GETNEXTITEM, (WPARAM)iSelItem, (LPARAM)LVNI_SELECTED);
	}

CLEANUP:
	if(pcItems)
		*pcItems = cItems;
	return cItems;
}


//////////////////////////////////////////////////////////////////
// LONG LV_GetAllItems
//
//////////////////////////////////////////////////////////////////
LONG LV_GetAllItems(HWND hWnd, ULONG* pcItems, LONG** prgItems, LONG** prgParams)
{
	//Get the total Items
	LONG i=0;
	LONG cItems = SendMessage(hWnd, LVM_GETITEMCOUNT, 0, 0);

	//Alloc Output Array
	if(prgItems)
		SAFE_ALLOC(*prgItems, LONG, cItems);
	if(prgParams)
		SAFE_ALLOC(*prgParams, LONG, cItems);

	//Find all Items
	for(i=0; i<cItems; i++)
	{
		if(prgItems)
			(*prgItems)[i] = i;
		if(prgParams)
			(*prgParams)[i] = LV_GetItemParam(hWnd, i, 0);
	}

CLEANUP:
	if(pcItems)
		*pcItems = cItems;
	return cItems;
}


//////////////////////////////////////////////////////////////////
// BOOL LV_FindItem
//
//////////////////////////////////////////////////////////////////
LONG LV_FindItem(HWND hWnd, CHAR* szName, LONG iStart)
{
	//LVM_FINDITEM
	LV_FINDINFO lvFindInfo = { LVFI_STRING, szName, 0, 0, 0};
	return SendMessage(hWnd, LVM_FINDITEM, (WPARAM)iStart, (LPARAM)&lvFindInfo);
}


////////////////////////////////////////////////////////////////
// LONG LV_FindItem
//
/////////////////////////////////////////////////////////////////
LONG LV_FindItem(HWND hWnd, LONG lParam, LONG iStart)
{
	//LVM_FINDITEM
	LV_FINDINFO lvFindInfo = { LVIF_PARAM, 0, lParam, 0, 0};
	return SendMessage(hWnd, LVM_FINDITEM, (WPARAM)iStart, (LPARAM)&lvFindInfo);
}


//////////////////////////////////////////////////////////////////
// BOOL TV_InsertItem
//
//////////////////////////////////////////////////////////////////
HTREEITEM TV_InsertItem(HWND hWnd, HTREEITEM hParent, HTREEITEM hInsAfter, CHAR* szName, LONG iParam, LONG iImage, LONG iSelectedImage)
{
	TV_INSERTSTRUCT tvInsertStruct = { hParent, hInsAfter, { TVIF_TEXT | TVIF_IMAGE | TVIF_SELECTEDIMAGE | TVIF_PARAM, 0, 0, 0, szName, 0, iImage, iSelectedImage, 0, iParam} };
	
	//TVM_INSERTITEM
	return (HTREEITEM)SendMessage(hWnd, TVM_INSERTITEM, 0, (LPARAM)&tvInsertStruct);
}


//////////////////////////////////////////////////////////////////
// BOOL TV_GetItemText
//
//////////////////////////////////////////////////////////////////
LONG TV_GetItemText(HWND hWnd, HTREEITEM hItem, CHAR* szBuffer, LONG ulMaxSize)
{
	TVITEM tvItem = { TVIF_TEXT, hItem, 0, 0, szBuffer, ulMaxSize, 0, 0, 0, 0};

	//GetItem Text
	return SendMessage(hWnd, TVM_GETITEM, 0, (LPARAM)&tvItem);
}


//////////////////////////////////////////////////////////////////
// BOOL TV_GetItemParam
//
//////////////////////////////////////////////////////////////////
LONG TV_GetItemParam(HWND hWnd, HTREEITEM hItem)
{
	//no-op
	if(hItem == NULL)
		return LVM_ERR;
	
	TVITEM tvItem = { TVIF_PARAM, hItem, 0, 0, NULL, 0, 0, 0, 0, 0};

	//GetItem
	SendMessage(hWnd, TVM_GETITEM, 0, (LPARAM)&tvItem);

	//return the lParam
	return tvItem.lParam;
}


//////////////////////////////////////////////////////////////////
// BOOL TV_FindItem
//
//////////////////////////////////////////////////////////////////
HTREEITEM TV_FindItem(HWND hWnd, HTREEITEM hParent, CHAR* szName)
{
	ASSERT(hWnd);
	ASSERT(szName);
	CHAR szBuffer[MAX_NAME_LEN];
	TVITEM tvItem = { TVIF_TEXT, 0, 0, 0, szBuffer, MAX_NAME_LEN, 0, 0, 0, 0};
	
	tvItem.hItem = (HTREEITEM)SendMessage(hWnd, TVM_GETNEXTITEM, (WPARAM)TVGN_CHILD, (LPARAM)hParent);
	while(tvItem.hItem)
	{
		//Try to find this string in the Tree
		if(SendMessage(hWnd, TVM_GETITEM, 0, (LPARAM)&tvItem))
		{
			if(strcmp(szName, tvItem.pszText)==0)
				return tvItem.hItem;
		}

		//Otherwise get the NextItem and continue...
		tvItem.hItem = (HTREEITEM)SendMessage(hWnd, TVM_GETNEXTITEM, (WPARAM)TVGN_NEXT, (LPARAM)tvItem.hItem);
	}

	return NULL;
}


//////////////////////////////////////////////////////////////////
// BOOL SB_SetScrollInfo
//
//////////////////////////////////////////////////////////////////
LONG SB_SetScrollInfo(HWND hWnd, INT dwFlags, INT iPos, INT iRangeSize, INT iPageSize, BOOL fRedraw)
{
	SCROLLINFO ScrollInfo = { sizeof(SCROLLINFO), SIF_ALL, 0, iRangeSize, iPageSize, iPos, 0};
	
	//SetScrollInfo
	return SetScrollInfo(hWnd, dwFlags, &ScrollInfo, fRedraw);
}


//////////////////////////////////////////////////////////////////
// BOOL SB_GetScrollInfo
//
//////////////////////////////////////////////////////////////////
LONG SB_GetScrollInfo(HWND hWnd, INT dwFlags, INT* piPos, INT* piRangeSize, INT* piPageSize)
{
	INT iPos = 0;
	INT iRangeSize = 0;
	INT iPageSize = 0;

	SCROLLINFO ScrollInfo = { sizeof(SCROLLINFO), SIF_ALL, 0, iRangeSize, iPageSize, iPos, 0};
	
	//SetScrollInfo
	INT iResult = GetScrollInfo(hWnd, dwFlags, &ScrollInfo);

	//Fill in arguments
	if(piPos)
		*piPos = iPos;
	if(piRangeSize)
		*piRangeSize = iRangeSize;
	if(piPageSize)
		*piPageSize = iPageSize;

	return iResult;
}

//////////////////////////////////////////////////////////////////
// HRESULT ReplaceString
//
//////////////////////////////////////////////////////////////////
HRESULT	ReplaceString(CHAR* pszBuffer, CHAR* pszTokens, CHAR* pszReplace)
{
	//no-op
	if(pszBuffer == NULL || pszTokens == NULL || pszReplace == NULL)
		return E_FAIL;

	//Make sure we have the same replacement as the search string
	//IE: (this function doesn't handle growing/shrinking the string)
	ULONG ulStrLen = strlen(pszTokens);
	if(ulStrLen != strlen(pszReplace) || ulStrLen > strlen(pszBuffer))
		return E_FAIL;
		
	//Repleace all occurrences of pszTokens with pszReplace
	CHAR* pszFound = strstr(pszBuffer, pszTokens);
	while(pszFound)
	{
		memcpy(pszFound, pszReplace, sizeof(CHAR)*ulStrLen);
		
		//Try to find the next occurrence
		pszFound += ulStrLen;
		pszFound = strstr(pszFound, pszTokens);
	}

	return S_OK;
}

//////////////////////////////////////////////////////////////////
// HRESULT GetEditBoxSelection
//
//////////////////////////////////////////////////////////////////
HRESULT	GetEditBoxSelection(HWND hWndEdit, CHAR* pszBuffer, ULONG ulMaxSize)
{
	ASSERT(hWndEdit);
	ASSERT(pszBuffer);
	ASSERT(ulMaxSize);

	LONG dwStartPos = 0;
	LONG dwEndPos = 0;
	LONG iCharIndex = -1;
	LONG iLineLength = 0;
	CHAR* psz = pszBuffer;

	//Obtain the EditBox text (rules below).
	//  1.  First try to obtain whatever text is highlighted (even multiline highlight)
	//  2.  Or if nothing is highlighted take the entire line of the cursor

	//Obtain the Selected Text Start and End Positions
	SendMessage(hWndEdit, EM_GETSEL, (WPARAM)&dwStartPos, (LPARAM)&dwEndPos);

	//If there is no selected Text
	//Just get the entire line, as if it were highlighted...
	if(dwStartPos == dwEndPos)
	{
		dwStartPos = SendMessage(hWndEdit, EM_LINEINDEX, (WPARAM)-1, (LPARAM)0);
		dwEndPos = dwStartPos + SendMessage(hWndEdit, EM_LINELENGTH, dwStartPos, 0);
	}
	
	//We actually may not have enough buffer for the selected text
	if(dwEndPos-dwStartPos > (LONG)ulMaxSize)
		dwEndPos = dwStartPos + ulMaxSize -1;
	
	//Get Line for the Current Position
	LONG iLine = SendMessage(hWndEdit, EM_LINEFROMCHAR, dwStartPos, 0);
	LONG iEndLine = SendMessage(hWndEdit, EM_LINEFROMCHAR, dwEndPos, 0);

	while(iLine <= iEndLine)
	{
		//Get the Text for this line			
		//The first word of the buffer represents the MaxSize of the Buffer 
		((DWORD*)psz)[0] = ulMaxSize-1;
		iLineLength = SendMessage(hWndEdit, EM_GETLINE, iLine, (LPARAM)psz);
		
		//Special Character Post Processing...
		ReplaceString(psz, "\t", " ");
		ReplaceString(psz, "\n", " ");
		ReplaceString(psz, "\r", " ");

		//The first time through there are fixups that need to occur
		//This is due to the fact that the starting hightlighted char may not
		//be the first char on the line.
		if(iCharIndex == -1)
		{
			//Obtain the CharIndex of the first char of the first line
			iCharIndex = SendMessage(hWndEdit, EM_LINEINDEX, (WPARAM)iLine, (LPARAM)0);
			iLineLength = iLineLength - (dwStartPos - iCharIndex);
			memmove(psz, psz + (dwStartPos - iCharIndex), iLineLength);
		}

		//Null Terminator
		if(iLine == iEndLine)
		{
			//Obtain the CharIndex of the first char of the last line
			iCharIndex = SendMessage(hWndEdit, EM_LINEINDEX, (WPARAM)iLine, (LPARAM)0);
			
			//Also have the other case where the startpos may be the same line 
			//as the endpos
			if(dwStartPos > iCharIndex)
				psz[dwEndPos-dwStartPos] = wEOL;
			else
				psz[dwEndPos-iCharIndex] = wEOL;
		}
		else
		{ 
			psz += iLineLength;

			//Since were dealing with MultiLine edit here
			//add an extra space in between statements, to make it more friendly...
			psz[0] = ' ';
			psz += 1;
		}
		
		//Get TheNext Line from the current position
		iLine++;
	}
	
	//Highlight the text we used...
	SendMessage(hWndEdit, EM_SETSEL, (WPARAM)dwStartPos, (LPARAM)dwEndPos);
	return S_OK;
}


//////////////////////////////////////////////////////////////////
// HRESULT ReplaceEditBoxSelection
//
//////////////////////////////////////////////////////////////////
HRESULT	ReplaceEditBoxSelection(HWND hWndEdit, CHAR* pszBuffer)
{
	ASSERT(hWndEdit);
	ASSERT(pszBuffer);

	LONG dwStartPos = 0;
	LONG dwStartPos2 = 0;
	LONG dwEndPos = 0;
	LONG dwEndPos2 = 0;

	//Obtain the Selected Text Start and End Positions
	SendMessage(hWndEdit, EM_GETSEL, (WPARAM)&dwStartPos, (LPARAM)&dwEndPos);

	//Replace the Selection
	SendMessage(hWndEdit, EM_REPLACESEL, TRUE, (LPARAM)pszBuffer);

	//Highlight the text we used...
	SendMessage(hWndEdit, EM_GETSEL, (WPARAM)&dwStartPos2, (LPARAM)&dwEndPos2);
	SendMessage(hWndEdit, EM_SETSEL, (WPARAM)dwStartPos, (LPARAM)dwEndPos2);
	return S_OK;
}


////////////////////////////////////////////////////////////////
// BOOL DisplayContextMenu
//
/////////////////////////////////////////////////////////////////
BOOL DisplayContextMenu(HINSTANCE hInst, HWND hWnd, WORD iID, INT xPos, INT yPos, HWND hWndParent, BOOL fRelCords)
{
	//Load the SubMenu
	HMENU hMenu = LoadMenu(hInst, MAKEINTRESOURCE(iID));
	HMENU hSubMenu = GetSubMenu(hMenu, 0);

	RECT rect;
	GetWindowRect(hWnd, &rect);

	//Coordinates might be Screen Coordinates or Relative
	if(fRelCords)
	{
		rect.top += yPos;	
		rect.bottom += yPos;	
		rect.left += xPos;	
		rect.right += xPos;	
	}

	//This message may have come from the keyboard
	//Just display at upper left corner
	if(xPos < rect.left || xPos > rect.right || yPos < rect.top || yPos > rect.bottom)
	{
		xPos = rect.left + 5;
		yPos = rect.top + 5;
	}
	
	//Display SubMenu
	//Wants (x,y) in Screen Coordinates
	BOOL bResult = TrackPopupMenu(hSubMenu,
          TPM_LEFTALIGN | TPM_RIGHTBUTTON,
          xPos,
          yPos,
          0,
          hWndParent,
          NULL);

	DestroyMenu(hSubMenu);
	DestroyMenu(hMenu);
	return bResult;
}


///////////////////////////////////////////////////////////////
// BrowseOpenFileName
//
///////////////////////////////////////////////////////////////
HRESULT BrowseOpenFileName(HINSTANCE hInstance, HWND hWnd, CHAR* pszTitle, CHAR* pszFileName, ULONG ulMaxSize)
{
	ASSERT(pszFileName);
	static ULONG ulFilterIndex = 5;  //Default to *.* 
	HRESULT hr = E_FAIL;

	CHAR szCustFilter[MAX_NAME_LEN];
	strcpy(szCustFilter, "User Defined");

	//Setup OPENFILENAME struct...
	OPENFILENAME    ofn;
	memset( &ofn, 0, sizeof( ofn ));
	ofn.lStructSize       = sizeof( OPENFILENAME );
	ofn.hwndOwner         = hWnd;
	ofn.hInstance         = hInstance;
	ofn.lpstrFile         = pszFileName;
	ofn.nMaxFile          = ulMaxSize;
	ofn.lpstrTitle		  = pszTitle;
	ofn.lpstrFilter		  = "DataSource Files (.dsn;.kag;.sav)\0*.dsn;*.kag;*.sav;\0DataBase Files (.mdb;.db;.dbf)\0*.mdb;*.db;*.dbf;\0Program Files (.xls;.clb)\0*.xls;*.clb;\0Text Files (.txt;.csv)\0*.txt;*.csv\0All Files (*.*)\0*.*\0\0";
	ofn.lpstrCustomFilter = szCustFilter;
	ofn.nMaxCustFilter	  = MAX_NAME_LEN;
	ofn.nFilterIndex	  = ulFilterIndex;
	ofn.Flags             = OFN_CREATEPROMPT	| 
							OFN_EXPLORER		|
							OFN_FILEMUSTEXIST	|
							OFN_PATHMUSTEXIST   |
							OFN_HIDEREADONLY	|
							OFN_OVERWRITEPROMPT	|
							OFN_SHAREAWARE;
	
	//Display Common Dialog to obtain File To Load...
	TESTC_(GetOpenFileName(&ofn), TRUE)
	
	//Save the Choosen Filter Index
	ulFilterIndex = ofn.nFilterIndex;
	hr = S_OK;

CLEANUP:
	return hr;
}


///////////////////////////////////////////////////////////////
// BrowseOpenFileName
//
///////////////////////////////////////////////////////////////
HRESULT BrowseOpenFileName(HINSTANCE hInstance, HWND hWnd, CHAR* pszTitle, WCHAR* pwszFileName, ULONG ulMaxSize)
{
	ASSERT(pwszFileName);
	HRESULT hr = E_FAIL;
	CHAR szBuffer[MAX_QUERY_LEN];
	szBuffer[0] = EOL;

	//Delegation
	TESTC(hr = BrowseOpenFileName(hInstance, hWnd, pszTitle, szBuffer, MAX_QUERY_LEN));

	//Now convert to WCHAR
	TESTC(hr = ConvertToWCHAR(szBuffer, pwszFileName, ulMaxSize));

CLEANUP:
	return hr;
}

						
///////////////////////////////////////////////////////////////
// BrowseSaveFileName
//
///////////////////////////////////////////////////////////////
HRESULT BrowseSaveFileName(HINSTANCE hInstance, HWND hWnd, CHAR* pszTitle, CHAR* pszFileName, ULONG ulMaxSize)
{
	HRESULT hr = E_FAIL;
	ASSERT(pszFileName);
	static ULONG ulFilterIndex = 5;  //Default to *.* 

	CHAR szCustFilter[MAX_NAME_LEN];
	strcpy(szCustFilter, "User Defined");

	//Setup OPENFILENAME struct...
	OPENFILENAME    ofn;
	memset( &ofn, 0, sizeof( ofn ));
	ofn.lStructSize       = sizeof( OPENFILENAME );
	ofn.hwndOwner         = hWnd;
	ofn.hInstance         = hInstance;
	ofn.lpstrFile         = pszFileName;
	ofn.nMaxFile          = ulMaxSize;
	ofn.lpstrTitle		  = pszTitle;
	ofn.lpstrFilter		  = "DataSource Files (.dsn;.kag;.sav)\0*.dsn;*.kag;*.sav;\0DataBase Files (.mdb;.db;.dbf)\0*.mdb;*.db;*.dbf;\0Program Files (.xls;.clb)\0*.xls;*.clb;\0Text Files (.txt;.csv)\0*.txt;*.csv\0All Files (*.*)\0*.*\0\0";
	ofn.lpstrCustomFilter = szCustFilter;
	ofn.nMaxCustFilter	  = MAX_NAME_LEN;
	ofn.nFilterIndex	  = ulFilterIndex;
	ofn.Flags             = OFN_CREATEPROMPT	| 
							OFN_EXPLORER		|
							OFN_PATHMUSTEXIST   |
							OFN_HIDEREADONLY	|
							OFN_OVERWRITEPROMPT	|
							OFN_SHAREAWARE;
	
	//Display Common Dialog to obtain File To Save...
	if(GetSaveFileName(&ofn))
	{
		ulFilterIndex = ofn.nFilterIndex;
		hr = S_OK;
	}

	return hr;
}


///////////////////////////////////////////////////////////////
// BrowseSaveFileName
//
///////////////////////////////////////////////////////////////
HRESULT BrowseSaveFileName(HINSTANCE hInstance, HWND hWnd, CHAR* pszTitle, WCHAR* pwszFileName, ULONG ulMaxSize)
{
	HRESULT hr = E_FAIL;
	ASSERT(pwszFileName);

	CHAR szBuffer[MAX_QUERY_LEN];
	szBuffer[0] = EOL;

	
	//Delegation
	if(SUCCEEDED(hr = BrowseSaveFileName(hInstance, hWnd, pszTitle, szBuffer, MAX_QUERY_LEN)))
	{
		hr = ConvertToWCHAR(szBuffer, pwszFileName, ulMaxSize);
	}

	return hr;
}


////////////////////////////////////////////////////////
// WCHAR* GetProgID
//
////////////////////////////////////////////////////////
WCHAR* GetProgID(CLSID clsid)
{
	WCHAR* pwszProgID = NULL;
	WCHAR wszBuffer[MAX_NAME_LEN];
	wszBuffer[0] = wEOL;

	//ProgID From the Sprecified CLSID
	if(FAILED(ProgIDFromCLSID(clsid, &pwszProgID)))
	{
		//If that does work, we will just return the 
		//String representation of the GUID
		StringFromGUID2(clsid, wszBuffer, MAX_NAME_LEN);
		pwszProgID = wcsDuplicate(wszBuffer);
	}

	return pwszProgID;
}


////////////////////////////////////////////////////////
// HRESULT GetRegEnumKey
//
////////////////////////////////////////////////////////
HRESULT GetRegEnumKey(HKEY hRootKey, CHAR* pszKeyName, DWORD dwIndex, CHAR* pszSubKeyName, ULONG cBytes)
{
	HRESULT hr = E_FAIL;
	HKEY hKey = NULL;

	//no-op, pszValueName is not required
	if(!pszKeyName || !pszSubKeyName || cBytes==0)
		return E_FAIL;
	
	//Obtain the Key for HKEY_CLASSES_ROOT\"SubKey"
	if(ERROR_SUCCESS != RegOpenKeyEx(hRootKey, pszKeyName, 0, KEY_READ, &hKey))
		goto CLEANUP;
	
	//Obtain the specified RegItem at the index specified
	if(ERROR_SUCCESS == RegEnumKey(hKey, dwIndex, pszSubKeyName, cBytes))
		hr = S_OK;
	
CLEANUP:
	if(hKey)
		RegCloseKey(hKey);
	
	return hr;
}


////////////////////////////////////////////////////////
// HRESULT GetRegEnumValue
//
////////////////////////////////////////////////////////
HRESULT GetRegEnumValue(HKEY hRootKey, CHAR* pszKeyName, DWORD dwIndex, CHAR* pszValueName, ULONG cBytes)
{
	HRESULT hr = E_FAIL;
	HKEY hKey = NULL;

	//no-op, pszValueName is not required
	if(!pszKeyName || !pszValueName || cBytes==0)
		return E_FAIL;
	
	//Obtain the Key for HKEY_CLASSES_ROOT\"SubKey"
	if(ERROR_SUCCESS != RegOpenKeyEx(hRootKey, pszKeyName, 0, KEY_READ, &hKey))
		goto CLEANUP;
	
	//Obtain the specified RegItem at the index specified
	if(ERROR_SUCCESS == RegEnumValue(hKey, dwIndex, pszValueName, &cBytes, 0, NULL, NULL, 0))
		hr = S_OK;
	
CLEANUP:
	if(hKey)
		RegCloseKey(hKey);
	
	return hr;
}


////////////////////////////////////////////////////////
// HRESULT GetRegEnumValue
//
////////////////////////////////////////////////////////
HRESULT GetRegEnumValue(HKEY hRootKey, CHAR* pszKeyName, DWORD dwIndex, WCHAR* pwszValueName, ULONG cBytes)
{
	CHAR szBuffer[MAX_QUERY_LEN];
	HRESULT hr = S_OK;

	TESTC(hr = GetRegEnumValue(hRootKey, pszKeyName, dwIndex, szBuffer, MAX_QUERY_LEN));
	TESTC(hr = ConvertToWCHAR(szBuffer, pwszValueName, cBytes));

CLEANUP:
	return hr;
}


////////////////////////////////////////////////////////
// HRESULT GetRegEntry
//
////////////////////////////////////////////////////////
HRESULT GetRegEntry(HKEY hRootKey, CHAR* pszKeyName, CHAR* pszValueName, CHAR* pszValue, ULONG cBytes)
{
	HRESULT hr = E_FAIL;
	HKEY hKey = NULL;

	//no-op, pszValueName is not required
	if(!pszKeyName || !pszValue || cBytes==0)
		return E_FAIL;
	
	//Obtain the Key for HKEY_CLASSES_ROOT\"SubKey"
	if(ERROR_SUCCESS != RegOpenKeyEx(hRootKey, pszKeyName, 0, KEY_READ, &hKey))
		goto CLEANUP;
	
	//Obtain the Data for the above key
	if(ERROR_SUCCESS == RegQueryValueEx(hKey, pszValueName, NULL, NULL, (LPBYTE)pszValue, &cBytes))
		hr = S_OK;
	
CLEANUP:
	if(hKey)
		RegCloseKey(hKey);
	
	return hr;
}


////////////////////////////////////////////////////////
// HRESULT GetRegEntry
//
////////////////////////////////////////////////////////
HRESULT GetRegEntry(HKEY hRootKey, CHAR* pszKeyName, CHAR* pszValueName, ULONG* pulValue)
{
	HRESULT hr = E_FAIL;
	HKEY hKey = NULL;
	ULONG cBytes = sizeof(ULONG);

	//no-op, pszValueName is not required
	if(!pszKeyName || !pulValue)
		return E_FAIL;
	
	//Obtain the Key for HKEY_CLASSES_ROOT\"SubKey"
	if(ERROR_SUCCESS != RegOpenKeyEx(hRootKey, pszKeyName, 0, KEY_READ, &hKey))
		goto CLEANUP;
	
	//Obtain the Data for the above key
	if(ERROR_SUCCESS == RegQueryValueEx(hKey, pszValueName, NULL, NULL, (LPBYTE)pulValue, &cBytes))
		hr = S_OK;
	
CLEANUP:
	if(hKey)
		RegCloseKey(hKey);
	
	return hr;
}

////////////////////////////////////////////////////////
// HRESULT GetRegEntry
//
////////////////////////////////////////////////////////
HRESULT GetRegEntry(HKEY hRootKey, CHAR* pszKeyName, CHAR* pszValueName, WCHAR* pwszValue, ULONG cBytes)
{
	CHAR szBuffer[MAX_NAME_LEN];
	HRESULT hr = S_OK;

	TESTC(hr = GetRegEntry(hRootKey, pszKeyName, pszValueName, szBuffer, MAX_NAME_LEN));
	TESTC(hr = ConvertToWCHAR(szBuffer, pwszValue, cBytes));

CLEANUP:
	return hr;
}


////////////////////////////////////////////////////////
// HRESULT SetRegEntry
//
////////////////////////////////////////////////////////
HRESULT SetRegEntry(HKEY hRootKey, CHAR* pszKeyName, CHAR* pszValueName, CHAR* pszValue)
{
	HRESULT hr = E_FAIL;
	HKEY hKey = NULL;
	ULONG dwDisposition;

	//no-op, pszValueName is not required
	if(!pszKeyName || !pszValue)
		return E_FAIL;
	
	//Create the Key for HKEY_CLASSES_ROOT\"SubKey"
    if(ERROR_SUCCESS != RegCreateKeyEx(hRootKey, pszKeyName, 0, NULL, 
			REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &hKey, &dwDisposition))
			goto CLEANUP;

	//Set the data for the above key
    if(ERROR_SUCCESS == RegSetValueEx(hKey, pszValueName, 0, REG_SZ,
            (BYTE*)pszValue, strlen(pszValue) + sizeof(CHAR)))
		hr = S_OK;

CLEANUP:
	if(hKey)
		RegCloseKey(hKey);
	
	return hr;
}


////////////////////////////////////////////////////////
// HRESULT SetRegEntry
//
////////////////////////////////////////////////////////
HRESULT SetRegEntry(HKEY hRootKey, CHAR* pszKeyName, CHAR* pszValueName, ULONG ulValue)
{
	HRESULT hr = E_FAIL;
	HKEY hKey = NULL;
	ULONG dwDisposition;

	//no-op, pszValueName is not required
	if(!pszKeyName)
		return E_FAIL;
	
	//Create the Key for HKEY_CLASSES_ROOT\"SubKey"
    if(ERROR_SUCCESS != RegCreateKeyEx(hRootKey, pszKeyName, 0, NULL, 
			REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &hKey, &dwDisposition))
			goto CLEANUP;

	//Set the data for the above key
    if(ERROR_SUCCESS == RegSetValueEx(hKey, pszValueName, 0, REG_DWORD,
            (LPBYTE)&ulValue, sizeof(ULONG)))
		hr = S_OK;

CLEANUP:
	if(hKey)
		RegCloseKey(hKey);
	
	return hr;
}


////////////////////////////////////////////////////////
// HRESULT SetRegEntry
//
////////////////////////////////////////////////////////
HRESULT SetRegEntry(HKEY hRootKey, CHAR* pszKeyName, CHAR* pszValueName, WCHAR* pwszValue)
{
	CHAR szBuffer[MAX_NAME_LEN];
	HRESULT hr = S_OK;
	
	TESTC(hr = ConvertToMBCS(pwszValue, szBuffer, MAX_NAME_LEN));
	TESTC(hr = SetRegEntry(hRootKey, pszKeyName, pszValueName, szBuffer));

CLEANUP:
	return hr;
}


////////////////////////////////////////////////////////
// HRESULT DelRegEntry
//
////////////////////////////////////////////////////////
HRESULT DelRegEntry(HKEY hRootKey, CHAR* pszKeyName)
{
	HKEY hKey = NULL;
	HRESULT hr;

	//no-op
	if(!pszKeyName)
		return E_FAIL;
	
	//Delete the Key for HKEY_CLASSES_ROOT\"SubKey"
	hr = RegDeleteKey(hRootKey, pszKeyName);

	//Entry successfully deleted - return S_OK
	if(hr==ERROR_SUCCESS) 
		return S_OK;

	//Entry not found - return S_FALSE
	if(hr==ERROR_FILE_NOT_FOUND)
		return S_FALSE;

	return E_FAIL;
}


///////////////////////////////////////////////////////////////
// Static Strings Messages
//
///////////////////////////////////////////////////////////////

extern WCHAR wsz_SUCCESS[]				= L"Microsoft OLE DB RowsetViewer - Success";
extern WCHAR wsz_WARNING[]				= L"Microsoft OLE DB RowsetViewer - Warning";
extern WCHAR wsz_INFO[]					= L"Microsoft OLE DB RowsetViewer - Info";
extern WCHAR wsz_ERROR[]				= L"Microsoft OLE DB RowsetViewer - Error";
extern WCHAR wsz_EXCEPTION[]			= L"Microsoft OLE DB RowsetViewer - Exception";
extern WCHAR wsz_ERRORINFO[]			= L"Microsoft OLE DB RowsetViewer - IErrorInfo";

//General String Values
extern WCHAR wsz_INVALID_VALUE_[]		= L"Invalid Value '%s' specified.  Please specify a value >= %lu and <= %lu.";

